/*
        Copyright  1995-2001, The AROS Development Team. All rights reserved.
        $Id: main.c,v 1.10 2008/08/13 10:17:11 tokai Exp $
 
        Desc:
        Lang: English
*/

/*********************************************************************************************/

#include "global.h"
#include "intuition/extensions.h"

#include <stdlib.h> /* for exit() */
#include <stdio.h>
#include <string.h>

#include "compilerspecific.h"
#include "debug.h"
//#include "arossupport.h"
#include "datatypes/soundclass.h" // for sound streaming

#define DB(x)  // kprintf x

extern struct NewMenu nm[];
extern struct NewMenu nmpict[];
extern struct NewMenu nmtext[];

/*********************************************************************************************/

/* Many datatype classes seem to rely on OM_NOTIFY calls coming back to the datatype object
   as OM_UPDATE :-\ */

#define BACK_CONNECTION 1

/*********************************************************************************************/

#define ARG_TEMPLATE    "FILE,CLIPBOARD/S,CLIPUNIT/K/N,SCREEN/S,PUBSCREEN/K,REQUESTER/S," \
"BOOKMARK/S,FONTNAME/K,FONTSIZE/K/N,BACKDROP/S,WINDOW/S," \
"PORTNAME/K,IMMEDIATE/S,REPEAT/S,PRTUNIT/K/N"

#define ARG_FILE        0
#define ARG_CLIPBOARD   1
#define ARG_CLIPUNIT    2
#define ARG_SCREEN      3
#define ARG_PUBSCREEN   4
#define ARG_REQUESTER   5
#define ARG_BOOKMARK    6
#define ARG_FONTNAME    7
#define ARG_FONTSIZE    8
#define ARG_BACKDROP    9
#define ARG_WINDOW      10
#define ARG_PORTNAME    11
#define ARG_IMMEDIATE   12
#define ARG_REPEAT      13
#define ARG_PRTUNIT     14

#define NUM_ARGS        15

#define INTUI50 (((struct Library *)IntuitionBase)->lib_Version >= 50)

/*********************************************************************************************/

static struct libinfo
{
        APTR        var;
        STRPTR      name;
        WORD        version;
}
libtable[] =
        {
                {&IntuitionBase     , "intuition.library"           , 39    },
                {&GfxBase           , "graphics.library"            , 39    },
                {&GadToolsBase      , "gadtools.library"            , 39    },
                {&LayersBase        , "layers.library"              , 39    },
                {&UtilityBase       , "utility.library"             , 39    },
                {&KeymapBase        , "keymap.library"              , 39    },
                {&DataTypesBase     , "datatypes.library"           , 39    },
                {&DiskfontBase      , "diskfont.library"            , 39    },
                {NULL                                                       }
        };

static struct TextAttr  textattr;
static struct TextFont  *font;
static struct RDArgs    *myargs;
static IPTR             args[NUM_ARGS];
static UBYTE            fontname[256];
static WORD             winwidth, winheight;
static WORD             sizeimagewidth, sizeimageheight;
static BOOL             model_has_members;

/*********************************************************************************************/

static void CloseLibs(void);
static void KillFont(void);
static void FreeArguments(void);
static void KillICObjects(void);
static void FreeVisual(void);
static void KillGadgets(void);
static void CloseDTO(void);
static void KillWindow(void);
static void ScrollTo(UWORD dir, UWORD quali);
static void FitToWindow(void);

/*********************************************************************************************/

WORD ShowMessage(CONST_STRPTR title, CONST_STRPTR text, CONST_STRPTR gadtext)
{
  if (text)
  {
    if (IntuitionBase && !((struct Process *)FindTask(NULL))->pr_CLI)
    {
        struct EasyStruct es;

        es.es_StructSize   = sizeof(es);
        es.es_Flags        = 0;
        es.es_Title        = (STRPTR) title;
        es.es_TextFormat   = (STRPTR) text;
        es.es_GadgetFormat = (STRPTR) gadtext;

        return EasyRequestArgs(win, &es, NULL, NULL);
    }
    else
    {
        printf("%s: %s\n", title, text);
    }
  }

  return 0;
}

void OutputMessage(CONST_STRPTR msg)
{
	ShowMessage("MultiView", msg, MSG(MSG_OK));
}

/*********************************************************************************************/

void Cleanup(CONST_STRPTR msg)
{
        ShowMessage("MultiView", msg, MSG(MSG_OK));

        KillWindow();
        KillMenus();
        KillGadgets();
        FreeVisual();
        CloseDTO();
        KillICObjects();
        KillFont();
        FreeArguments();
        CloseLibs();
        CleanupLocale();

        exit(prog_exitcode);
}

/*********************************************************************************************/

static void OpenLibs(void)
{
        struct libinfo *li;

        for(li = libtable; li->var; li++)
        {
                if (!((*(struct Library **)li->var) = OpenLibrary(li->name, li->version)))
                {
                        sprintf(s, MSG(MSG_CANT_OPEN_LIB), li->name, li->version);
                        Cleanup(s);
                }
        }

}

/*********************************************************************************************/

static void CloseLibs(void)
{
        struct libinfo *li;

        for(li = libtable; li->var; li++)
        {
                if (*(struct Library **)li->var) CloseLibrary((*(struct Library **)li->var));
        }
}

/*********************************************************************************************/

static void LoadFont(void)
{
        font = OpenDiskFont(&textattr);
        if (!font)
        {
                textattr.ta_Name  = "topaz.font";
                textattr.ta_YSize = 8;
                textattr.ta_Style = 0;
                textattr.ta_Flags = 0;

                font = OpenFont(&textattr);
        }
}

/*********************************************************************************************/

static void KillFont(void)
{
        if (font) CloseFont(font);
}

/*********************************************************************************************/

static void InitDefaults(void)
{
        struct TextFont *defaultfont = GfxBase->DefaultFont;

        /* This might be a bit problematic depending on how default system font
           switching through Font prefs program works and if then the previous
           default system font is closed or not. So this is very likely only safe
           when in such a case the previous font is not closed (means -> the font
           will remain in memory in any case)

           ClipView example program on Amiga Dev CD also does it like this. So ... */

        textattr.ta_Name  = defaultfont->tf_Message.mn_Node.ln_Name;
        textattr.ta_YSize = defaultfont->tf_YSize;
        textattr.ta_Style = defaultfont->tf_Style;
        textattr.ta_Flags = defaultfont->tf_Flags;
}

static void GetArguments(int argc, char **argv)
{
        if (argc)
        {
                /*
                 * Starting from shell
                 */

                if (!(myargs = ReadArgs(ARG_TEMPLATE, args, NULL)))
                {
                        Fault(IoErr(), 0, s, sizeof(s));
                        Cleanup(s);
                }

                filename = (STRPTR)args[ARG_FILE];
                if (!filename && !args[ARG_CLIPBOARD])
                {
                        filename = GetFile();
                        if (!filename) Cleanup(NULL);
                }

                if (args[ARG_FONTNAME])
                {
                        stccpy(fontname, (char *)args[ARG_FONTNAME], sizeof(fontname) - 5);
                        if (!strstr(fontname, ".font")) strcat(fontname, ".font");

                        textattr.ta_Name = fontname;
                }

                if (args[ARG_FONTSIZE])
                {
                        textattr.ta_YSize = *(LONG *)args[ARG_FONTSIZE];
                }

				if (args[ARG_PUBSCREEN])
				{
					stccpy(pubscrname, (char *)args[ARG_PUBSCREEN], sizeof(pubscrname) - 2);
				}
        }
        else
        {
                /*
                 * Starting from WB. This part is probably
                 * not complete. Feel free to fix [zapek]
                 *
                 * Probably multiargs should do something
                 * smart here... - Piru
                 */
                struct WBStartup *wbs = (struct WBStartup *)argv;
                struct WBArg *wba = wbs->sm_ArgList;

                if (wbs->sm_NumArgs > 1)
                {
                        wba++;

                        if (wba->wa_Lock)
                        {
                                if ((filename = GetFileFromLock(wba->wa_Lock)))
                                {
                                        if (!AddPart(filename, wba->wa_Name, 300)) /* XXX: yeah yeah.. that 300.. */
                                                Cleanup(NULL);
                                }
                        }
                }
                else
                {
                        /* if no arg is given, use filerequester - Piru */

                        filename = GetFile();
                        if (!filename) Cleanup(NULL);
                }
        }
}

/*********************************************************************************************/

static void FreeArguments(void)
{
        if (myargs) FreeArgs(myargs);
}

/*********************************************************************************************/

static void MakeICObjects(void)
{
        static const struct TagItem dto_to_vert_map[] =
            {
                    {DTA_TopVert            , PGA_Top       },
                    {DTA_VisibleVert        , PGA_Visible   },
                    {DTA_TotalVert          , PGA_Total     },
                    {TAG_DONE                               }
            };
        static const struct TagItem dto_to_horiz_map[] =
            {
                    {DTA_TopHoriz           , PGA_Top       },
                    {DTA_VisibleHoriz       , PGA_Visible   },
                    {DTA_TotalHoriz         , PGA_Total     },
                    {TAG_DONE                               }
            };
        static const struct TagItem vert_to_dto_map[] =
            {
                    {PGA_Top                , DTA_TopVert   },
                    {TAG_DONE                               }
            };
        static const struct TagItem horiz_to_dto_map[] =
            {
                    {PGA_Top                , DTA_TopHoriz  },
                    {TAG_DONE                               }
            };

        model_obj           = NewObject(NULL, MODELCLASS, ICA_TARGET, ICTARGET_IDCMP,
                                        TAG_DONE);
        dto_to_vert_ic_obj  = NewObject(NULL, ICCLASS, ICA_MAP, (ULONG)dto_to_vert_map,
                                        TAG_DONE);
        dto_to_horiz_ic_obj = NewObject(NULL, ICCLASS, ICA_MAP, (ULONG)dto_to_horiz_map,
                                        TAG_DONE);
        vert_to_dto_ic_obj  = NewObject(NULL, ICCLASS, ICA_MAP, (ULONG)vert_to_dto_map,
                                        TAG_DONE);
        horiz_to_dto_ic_obj = NewObject(NULL, ICCLASS, ICA_MAP, (ULONG)horiz_to_dto_map,
                                        TAG_DONE);
#if BACK_CONNECTION
        model_to_dto_ic_obj = NewObject(NULL, ICCLASS, TAG_DONE);
#endif

        if (!model_obj                  ||
                !dto_to_vert_ic_obj     ||
                !dto_to_horiz_ic_obj    ||
                !vert_to_dto_ic_obj     ||
                !horiz_to_dto_ic_obj
#if BACK_CONNECTION
                || !model_to_dto_ic_obj
#endif
           )
        {
                Cleanup(MSG(MSG_CANT_CREATE_IC));
        }

        DoMethod(model_obj, OM_ADDMEMBER, (ULONG)dto_to_vert_ic_obj);
        DoMethod(model_obj, OM_ADDMEMBER, (ULONG)dto_to_horiz_ic_obj);
#if BACK_CONNECTION
        DoMethod(model_obj, OM_ADDMEMBER, (ULONG)model_to_dto_ic_obj);
#endif

        model_has_members = TRUE;

}

/*********************************************************************************************/

static void KillICObjects(void)
{
        if (!model_has_members)
        {
                if (dto_to_vert_ic_obj) DisposeObject(dto_to_vert_ic_obj);
                if (dto_to_horiz_ic_obj) DisposeObject(dto_to_horiz_ic_obj);
#if BACK_CONNECTION
                if (model_to_dto_ic_obj) DisposeObject(model_to_dto_ic_obj);
#endif
        }

        if (model_obj) DisposeObject(model_obj);
        if (vert_to_dto_ic_obj) DisposeObject(vert_to_dto_ic_obj);
        if (horiz_to_dto_ic_obj) DisposeObject(horiz_to_dto_ic_obj);
}

/*********************************************************************************************/

static void GetVisual(void)
{
		STRPTR screenname = pubscrname[0] != '\0' ? pubscrname : NULL;

        scr = LockPubScreen(screenname);
        if (!scr)
		{
			if (screenname)
			{
				scr = LockPubScreen(NULL);
				if (!scr)
				{
					Cleanup(MSG(MSG_CANT_LOCK_SCR));
				}
			}
			else
			{
				Cleanup(MSG(MSG_CANT_LOCK_SCR));
			}
		}

        dri = GetScreenDrawInfo(scr);
        if (!dri) Cleanup(MSG(MSG_CANT_GET_DRI));

        vi = GetVisualInfoA(scr, NULL);
        if (!vi) Cleanup(MSG(MSG_CANT_GET_VI));
}

/*********************************************************************************************/

static void FreeVisual(void)
{
        if (vi) FreeVisualInfo(vi);
        if (dri) FreeScreenDrawInfo(scr, dri);
        if (scr) UnlockPubScreen(NULL, scr);
}

/*********************************************************************************************/

static void MakeGadgets(void)
{
        static WORD img2which[] =
            {
                UPIMAGE,
                DOWNIMAGE,
                LEFTIMAGE,
                RIGHTIMAGE,
                SIZEIMAGE
            };

        IPTR imagew[NUM_IMAGES], imageh[NUM_IMAGES];
        WORD v_offset, h_offset, btop, i;

        for(i = 0; i < NUM_IMAGES; i++)
        {
                img[i] = NewObject(NULL, SYSICLASS, SYSIA_DrawInfo      , (ULONG)dri           ,
                                   SYSIA_Which         , (ULONG)img2which[i]  ,
                                   TAG_DONE);

                if (!img[i]) Cleanup(MSG(MSG_CANT_CREATE_SYSIMAGE));

                GetAttr(IA_Width,(Object *)img[i],&imagew[i]);
                GetAttr(IA_Height,(Object *)img[i],&imageh[i]);
        }

        sizeimagewidth  = imagew[IMG_SIZE];
        sizeimageheight = imageh[IMG_SIZE];

        btop = scr->WBorTop + dri->dri_Font->tf_YSize + 1;

        v_offset = imagew[IMG_DOWNARROW] / 4;
        h_offset = imageh[IMG_LEFTARROW] / 4;

        gad[GAD_UPARROW] = NewObject(NULL, BUTTONGCLASS,
                                     GA_Image            , (ULONG)img[IMG_UPARROW]                                                      ,
                                     GA_RelRight         , -imagew[IMG_UPARROW] + 1                                              ,
                                     GA_RelBottom        , -imageh[IMG_DOWNARROW] - imageh[IMG_UPARROW] - imageh[IMG_SIZE] + 1   ,
                                     GA_ID               , GAD_UPARROW                                                           ,
                                     GA_RightBorder      , TRUE                                                                  ,
                                     GA_Immediate        , TRUE                                                                  ,
                                     GA_RelVerify        , TRUE                                          ,
                                     TAG_DONE);

        gad[GAD_DOWNARROW] = NewObject(NULL, BUTTONGCLASS,
                                       GA_Image            , (IPTR) img[IMG_DOWNARROW]                            ,
                                       GA_RelRight         , -imagew[IMG_UPARROW] + 1                      ,
                                       GA_RelBottom        , -imageh[IMG_UPARROW] - imageh[IMG_SIZE] + 1   ,
                                       GA_ID               , GAD_DOWNARROW                                 ,
                                       GA_RightBorder      , TRUE                                          ,
                                       GA_Previous         , (IPTR) gad[GAD_UPARROW]                              ,
                                       GA_Immediate        , TRUE                                          ,
                                       GA_RelVerify        , TRUE                                          ,
                                       TAG_DONE);

        gad[GAD_VERTSCROLL] = NewObject(NULL, PROPGCLASS,
                                        GA_Top              , btop + 1                                                                      ,
                                        GA_RelRight         , INTUI50 ? (-GetSkinInfoAttr(dri,SI_RightPropWidth,0) - 1) : (-imagew[IMG_DOWNARROW] + v_offset + 1),
                                        GA_Width            , INTUI50 ? (GetSkinInfoAttr(dri,SI_RightPropWidth,0)) : (imagew[IMG_DOWNARROW] - v_offset * 2),
                                        GA_RelHeight        , -imageh[IMG_DOWNARROW] - imageh[IMG_UPARROW] - imageh[IMG_SIZE] - btop -2     ,
                                        GA_ID               , GAD_VERTSCROLL                                                                ,
                                        GA_Previous         , (IPTR) gad[GAD_DOWNARROW]                                                            ,
                                        GA_RightBorder      , TRUE                                                                          ,
                                        GA_RelVerify        , TRUE                                                                          ,
                                        GA_Immediate        , TRUE                                                                          ,
                                        PGA_NewLook         , TRUE                                                                          ,
                                        PGA_Borderless      , TRUE                                                                          ,
                                        PGA_Total           , 100                                                                           ,
                                        PGA_Visible         , 100                                                                           ,
                                        PGA_Freedom         , FREEVERT                                                                      ,
                                        PGA_NotifyBehaviour , PG_BEHAVIOUR_NICE                                                             ,
                                        TAG_DONE);

        gad[GAD_RIGHTARROW] = NewObject(NULL, BUTTONGCLASS,
                                        GA_Image            , (IPTR) img[IMG_RIGHTARROW]                           ,
                                        GA_RelRight         , -imagew[IMG_SIZE] - imagew[IMG_RIGHTARROW] + 1,
                                        GA_RelBottom        , -imageh[IMG_RIGHTARROW] + 1                   ,
                                        GA_ID               , GAD_RIGHTARROW                                ,
                                        GA_BottomBorder     , TRUE                                          ,
                                        GA_Previous         , (IPTR) gad[GAD_VERTSCROLL]                           ,
                                        GA_Immediate        , TRUE                                          ,
                                        GA_RelVerify        , TRUE                                          ,
                                        TAG_DONE);

        gad[GAD_LEFTARROW] = NewObject(NULL, BUTTONGCLASS,
                                       GA_Image            , (IPTR) img[IMG_LEFTARROW]                                                    ,
                                       GA_RelRight         , -imagew[IMG_SIZE] - imagew[IMG_RIGHTARROW] - imagew[IMG_LEFTARROW] + 1,
                                       GA_RelBottom        , -imageh[IMG_RIGHTARROW] + 1                                           ,
                                       GA_ID               , GAD_LEFTARROW                                                         ,
                                       GA_BottomBorder     , TRUE                                                                  ,
                                       GA_Previous         , (IPTR) gad[GAD_RIGHTARROW]                                                   ,
                                       GA_Immediate        , TRUE                                                                  ,
                                       GA_RelVerify        , TRUE                                                                  ,
                                       TAG_DONE);

        gad[GAD_HORIZSCROLL] = NewObject(NULL,PROPGCLASS,
                                         GA_Left             , scr->WBorLeft                                                                          ,
                                         GA_RelBottom        , INTUI50 ? (-GetSkinInfoAttr(dri,SI_BottomPropHeight,0) - 1) : (-imageh[IMG_LEFTARROW] + h_offset + 1),
                                         GA_RelWidth         , -imagew[IMG_LEFTARROW] - imagew[IMG_RIGHTARROW] - imagew[IMG_SIZE] - scr->WBorRight - 2,
                                         GA_Height           , INTUI50 ? (GetSkinInfoAttr(dri,SI_BottomPropHeight,0)) : (imageh[IMG_LEFTARROW] - (h_offset * 2)),
                                         GA_ID               , GAD_HORIZSCROLL                                                                        ,
                                         GA_Previous         , (IPTR) gad[GAD_LEFTARROW]                                                                     ,
                                         GA_BottomBorder     , TRUE                                                                                   ,
                                         GA_RelVerify        , TRUE                                                                                   ,
                                         GA_Immediate        , TRUE                                                                                   ,
                                         PGA_NewLook         , TRUE                                                                                   ,
                                         PGA_Borderless      , TRUE                                                                                   ,
                                         PGA_Total           , 100                                                                                    ,
                                         PGA_Visible         , 100                                                                                    ,
                                         PGA_Freedom         , FREEHORIZ                                                                              ,
                                         PGA_NotifyBehaviour , PG_BEHAVIOUR_NICE                                                                         ,
                                         TAG_DONE);

        for(i = 0;i < NUM_GADGETS;i++)
        {
                if (!gad[i]) Cleanup(MSG(MSG_CANT_CREATE_GADGET));
        }

        SetAttrs(gad[GAD_VERTSCROLL] , ICA_TARGET, (IPTR)vert_to_dto_ic_obj, TAG_DONE);
        SetAttrs(gad[GAD_HORIZSCROLL], ICA_TARGET, (IPTR)horiz_to_dto_ic_obj, TAG_DONE);
        SetAttrs(dto_to_vert_ic_obj  , ICA_TARGET, (IPTR)gad[GAD_VERTSCROLL], TAG_DONE);
        SetAttrs(dto_to_horiz_ic_obj , ICA_TARGET, (IPTR)gad[GAD_HORIZSCROLL], TAG_DONE);
}

/*********************************************************************************************/

static void KillGadgets(void)
{
        WORD i;

        for(i = 0; i < NUM_GADGETS;i++)
        {
                if (win) RemoveGadget(win, (struct Gadget *)gad[i]);
                if (gad[i]) DisposeObject(gad[i]);
                gad[i] = 0;
        }

        for(i = 0; i < NUM_IMAGES;i++)
        {
                if (img[i]) DisposeObject(img[i]);
                img[i] = NULL;
        }
}

/*********************************************************************************************/

void AddDTOToWin(void)
{
        EraseRect(win->RPort, win->BorderLeft,
                  win->BorderTop,
                  win->Width - 1 - win->BorderRight,
                  win->Height - 1 - win->BorderBottom);

        SetDTAttrs (dto, NULL, NULL, GA_Left        , win->BorderLeft + 2                           ,
                    GA_Top         , win->BorderTop + 2                            ,
                    GA_RelWidth    , - win->BorderLeft - win->BorderRight - 4      ,
                    GA_RelHeight   , - win->BorderTop - win->BorderBottom - 4      ,
                    TAG_DONE);

        AddDTObject(win, NULL, dto, -1);
#if 0
        RefreshDTObjects(dto, win, NULL, NULL);
#endif

}

/*********************************************************************************************/

static void OpenDTO(void)
{
        struct DTMethod *triggermethods;
        ULONG           *methods;
        STRPTR          objname = NULL;
        IPTR            val;
        struct DataType *dt;

        old_dto = dto;

        do
        {
                if (!old_dto && args[ARG_CLIPBOARD])
                {
                        APTR clipunit = 0;

                        if (args[ARG_CLIPUNIT]) clipunit = *(APTR *)args[ARG_CLIPUNIT];

                        D(bug("MultiView: calling NewDTObject\n"));

                        dto = NewDTObject(clipunit, ICA_TARGET    , (IPTR)model_obj,
                                          GA_ID         , 1000           ,
                                          DTA_SourceType, DTST_CLIPBOARD ,
                                          DTA_TextAttr  , (IPTR)&textattr,
                                          TAG_DONE);

                        D(bug("MultiView: NewDTObject returned %x\n", dto));
                }
                else
                {
                        dto = NewDTObject(filename, ICA_TARGET      , (IPTR)model_obj,
                                          GA_ID           , 1000           ,
                                          DTA_TextAttr    , (IPTR)&textattr,
                                          SDTA_Mode       , SDTA_Mode_Extended,  // streaming for sounds
                                          TAG_DONE);
                }

                if (!dto)
                {
                        ULONG errnum = IoErr();

                        if (errnum == DTERROR_UNKNOWN_DATATYPE)
                        {
                                BPTR lock = Lock(filename,ACCESS_READ);
                                if (lock)
                                {
                                        struct DataType *dtn;
                                        if ((dtn = ObtainDataTypeA(DTST_FILE, (APTR)lock, NULL)))
                                        {
                                                if (!Stricmp(dtn->dtn_Header->dth_Name, "directory"))
                                                {
                                                        UBYTE *newfilename;
                                                        int len;

                                                        /* file is a directory and no directory.datatype is installed */
                                                        stccpy(filenamebuffer, (filename ? filename : (STRPTR)""), sizeof(filenamebuffer)-1);
                                                        len = strlen(filenamebuffer);

                                                        if (len &&
                                                            filenamebuffer[len-1] != ':' &&
                                                            filenamebuffer[len-1] != '/')
                                                        {
                                                                strcat(filenamebuffer,"/");
                                                        }

                                                        newfilename = GetFile();
                                                        if (newfilename)
                                                        {
                                                                filename = newfilename;
                                                                continue;
                                                        }
                                                        else if (!old_dto)
                                                        {
                                                                errnum = -666;
                                                        }
                                                }
                                                ReleaseDataType(dtn);
                                        }
                                        UnLock(lock);
                                }
                        }

                        if ((LONG)errnum != -666)
                        {
                                if (errnum >= DTERROR_UNKNOWN_DATATYPE)
                                {
                                        sprintf(s, GetDTString(errnum), filename);
                                }
                                else
                                {
                                        Fault(errnum, 0, s, sizeof(s));
                                }
                        }

                        if (!old_dto) Cleanup(errnum == -666 ? NULL : s);
                        dto = old_dto;
                        return;
                }
        }
        while (!dto);

        stccpy(filenamebuffer, (filename ? filename : (STRPTR)""), sizeof(filenamebuffer));

        SetAttrs(vert_to_dto_ic_obj, ICA_TARGET, (IPTR)dto, TAG_DONE);
        SetAttrs(horiz_to_dto_ic_obj, ICA_TARGET, (IPTR)dto, TAG_DONE);
#if BACK_CONNECTION
        SetAttrs(model_to_dto_ic_obj, ICA_TARGET, (IPTR)dto, TAG_DONE);
#endif

        val = 0;
        GetDTAttrs(dto, DTA_NominalHoriz, (IPTR)&val, TAG_DONE);
        pdt_origwidth = winwidth  = (WORD)val;
        GetDTAttrs(dto, DTA_NominalVert , (IPTR)&val, TAG_DONE);
        pdt_origheight = winheight = (WORD)val;
        pdt_zoom = 1;
        pdt_fit_win = FALSE;
        pdt_keep_aspect = FALSE;

        /*
         *  Add 4 Pixels for border around DataType-Object
         *  See AddDTOToWin() for details
         */
        if(winwidth)
        {
                winwidth += 4;
        }

        if(winheight)
        {
                winheight += 4;
        }

        GetDTAttrs(dto, DTA_ObjName, (IPTR)&objname, TAG_DONE);
        stccpy(objnamebuffer, objname ? objname : filenamebuffer, sizeof(filenamebuffer));

        dt = NULL;
        dto_subclass_gid = 0;
        if (GetDTAttrs(dto, DTA_DataType, (IPTR)&dt, TAG_DONE))
        {
            if (dt)
            {
                dto_subclass_gid = dt->dtn_Header->dth_GroupID;
            }
        }

        dto_supports_write = FALSE;
        dto_supports_write_iff = FALSE;
        dto_supports_print = FALSE;
        dto_supports_copy = FALSE;
        dto_supports_selectall = FALSE;
        dto_supports_clearselected = FALSE;

        if (DoWriteMethod(NULL, DTWM_RAW)) dto_supports_write = TRUE;	/* probe raw saving */

        if ((methods = GetDTMethods(dto)))
        {
                if (FindMethod(methods, DTM_WRITE)) dto_supports_write_iff = TRUE;
                if (FindMethod(methods, DTM_PRINT)) dto_supports_print = TRUE;
                if (FindMethod(methods, DTM_COPY)) dto_supports_copy = TRUE;
                if (FindMethod(methods, DTM_SELECT)) dto_supports_selectall = TRUE;
                if (FindMethod(methods, DTM_CLEARSELECTED)) dto_supports_clearselected = TRUE;
        }

        dto_supports_contents =  FALSE;
        dto_supports_index =  FALSE;
        dto_supports_activate_field =  FALSE;
        dto_supports_next_field =  FALSE;
        dto_supports_prev_field =  FALSE;
        dto_supports_retrace =  FALSE;
        dto_supports_browse_next =  FALSE;
        dto_supports_browse_prev =  FALSE;
        dto_supports_search =  FALSE;
        dto_supports_search_next =  FALSE;
        dto_supports_search_prev =  FALSE;

        if ((triggermethods = (struct DTMethod *)GetDTTriggerMethods(dto)))
        {
                if (FindTriggerMethod(triggermethods, NULL, STM_CONTENTS))       dto_supports_contents       = TRUE;
                if (FindTriggerMethod(triggermethods, NULL, STM_INDEX))          dto_supports_index          = TRUE;
                if (FindTriggerMethod(triggermethods, NULL, STM_ACTIVATE_FIELD)) dto_supports_activate_field = TRUE;
                if (FindTriggerMethod(triggermethods, NULL, STM_NEXT_FIELD))     dto_supports_next_field     = TRUE;
                if (FindTriggerMethod(triggermethods, NULL, STM_PREV_FIELD))     dto_supports_prev_field     = TRUE;
                if (FindTriggerMethod(triggermethods, NULL, STM_RETRACE))        dto_supports_retrace        = TRUE;
                if (FindTriggerMethod(triggermethods, NULL, STM_BROWSE_NEXT))    dto_supports_browse_next    = TRUE;
                if (FindTriggerMethod(triggermethods, NULL, STM_BROWSE_PREV))    dto_supports_browse_prev    = TRUE;
                if (FindTriggerMethod(triggermethods, NULL, STM_SEARCH))         dto_supports_search         = TRUE;
                if (FindTriggerMethod(triggermethods, NULL, STM_SEARCH_NEXT))    dto_supports_search_next    = TRUE;
                if (FindTriggerMethod(triggermethods, NULL, STM_SEARCH_PREV))    dto_supports_search_prev    = TRUE;
        }

        D(bug("\nMultiview: Found Methods:%s%s%s%s%s%s\n",
              dto_supports_write ? " DTM_WRITE->RAW" : "",
              dto_supports_write_iff ? " DTM_WRITE->IFF" : "",
              dto_supports_print ? " DTM_PRINT" : "",
              dto_supports_copy ? " DTM_COPY" : "",
              dto_supports_selectall ? " DTM_SELECT" : "",
              dto_supports_clearselected ? " DTM_CLEARSELECTED" : ""));

        D(bug("Multiview: Found Triggers:%s%s%s%s%s%s%s\n\n",
              dto_supports_contents ? " STM_CONTENTS" : "",
              dto_supports_index ? " STM_INDEX" : "",
              dto_supports_activate_field ? " STM_ACTIVATE_FIELD" : "",
              dto_supports_next_field ? " STM_NEXT_FIELD" : "",
              dto_supports_prev_field ? " STM_PREV_FIELD" : "",
              dto_supports_retrace ? " STM_RETRACE" : "",
              dto_supports_browse_next ? " STM_BROWSE_NEXT" : "",
              dto_supports_browse_prev ? " STM_BROWSE_PREV" : "",
              dto_supports_search ? " STM_SEARCH" : "",
              dto_supports_search_next ? " STM_SEARCH_NEXT" : "",
              dto_supports_search_prev ? " STM_SEARCH_PREV" : ""));

        if (old_dto)
        {
                if (win) RemoveDTObject(win, old_dto);
                DisposeDTObject(old_dto);

                if (win)
                {
                        AddDTOToWin();
                        SetWindowTitles(win, objnamebuffer, (UBYTE *)~0);
                        SetMenuFlags();
                }
        }

}

/*********************************************************************************************/

static void CloseDTO(void)
{
        if (dto)
        {
                if (win) RemoveDTObject(win, dto);
                DisposeDTObject(dto);
                dto = NULL;
        }
}

/*********************************************************************************************/

static void MakeWindow(void)
{
        WORD minwidth, minheight, top, wintop, winbottom, winright, winleft;

        if (INTUI50)
        {
                top = GetSkinInfoAttr(dri,SI_ScreenTitlebarHeight,0);
                wintop = GetSkinInfoAttr(dri,SI_BorderTopTitle,0);
                winbottom = GetSkinInfoAttr(dri,SI_BorderBottomSize,0);
                winright = GetSkinInfoAttr(dri,SI_BorderRightSize,0);
                winleft = GetSkinInfoAttr(dri,SI_BorderLeft,0);
        } else {
                top = scr->BarHeight + 1;
                wintop = scr->WBorTop + scr->Font->ta_YSize + 1;
                winbottom = sizeimageheight;
                winright = sizeimagewidth;
                winleft = scr->WBorLeft;
        }

        if (!winwidth) winwidth = scr->Width - winright - winleft;
        if (!winheight) winheight = scr->Height - top - wintop - winbottom;

        minwidth  = ((winwidth  < 50) ? (winleft + winwidth + winright) : (winleft + 50 + winright));
        minheight = ((winheight < 50) ? (wintop + winheight + winbottom) : (wintop + 50 + winbottom));

        win = OpenWindowTags(0, WA_PubScreen     , (IPTR)scr          ,
                             WA_Title            , (IPTR)objnamebuffer   ,
                             WA_CloseGadget      , TRUE                  ,
                             WA_DepthGadget      , TRUE                  ,
                             WA_DragBar          , TRUE                  ,
                             WA_SizeGadget       , TRUE                  ,
                             WA_Activate         , TRUE                  ,
                             WA_SimpleRefresh    , TRUE                  ,
                           //WA_NoCareRefresh    , TRUE                  ,
                             WA_NewLookMenus     , TRUE                  ,
                             WA_Left             , 0                     ,
                             WA_Top              , top                   ,
                             WA_InnerWidth       , winwidth              ,
                             WA_InnerHeight      , winheight             ,
                             WA_AutoAdjust       , TRUE                  ,
                             WA_MinWidth         , minwidth              ,
                             WA_MinHeight        , minheight             ,
                             WA_MaxWidth         , 16383                 ,
                             WA_MaxHeight        , 16383                 ,
                             WA_Gadgets          , (IPTR)gad[GAD_UPARROW],
                             WA_IDCMP            , IDCMP_CLOSEWINDOW   |
                                                   IDCMP_GADGETUP      |
                                                   IDCMP_GADGETDOWN    |
                                                   IDCMP_MOUSEMOVE     |
                                                   IDCMP_VANILLAKEY    |
                                                   IDCMP_RAWKEY        |
                                                   IDCMP_IDCMPUPDATE   |
                                                   IDCMP_MENUPICK      |
                                                   IDCMP_NEWSIZE       |
                                                   IDCMP_REFRESHWINDOW |
                                                   IDCMP_INTUITICKS      ,
                   INTUI50 ? WA_SkinInfo : TAG_IGNORE, TRUE              ,
                             TAG_DONE);

        if (!win) Cleanup(MSG(MSG_CANT_CREATE_WIN));

        AddDTOToWin();

        SetMenuStrip(win, menus);
}

/*********************************************************************************************/

static void KillWindow(void)
{
        if (win)
        {
                if (dto) RemoveDTObject(win, dto);
                if (menus) ClearMenuStrip(win);
                CloseWindow(win);
                win = NULL;

                winwidth = winheight = 0;
        }
}

/*********************************************************************************************/

static void ScrollTo(UWORD dir, UWORD quali)
{
        IPTR val;
        LONG oldtop, top, total, visible, delta = 1;
        BOOL horiz;
        BOOL inc;

        switch(dir)
        {
        case RAWKEY_NM_WHEEL_UP:
                dir = CURSORUP;
                delta = 3;
                break;

        case RAWKEY_NM_WHEEL_DOWN:
                dir = CURSORDOWN;
                delta = 3;
                break;

        case RAWKEY_NM_WHEEL_LEFT:
                dir = CURSORLEFT;
                delta = 3;
                break;

        case RAWKEY_NM_WHEEL_RIGHT:
                dir = CURSORRIGHT;
                delta = 3;
                break;
        }

        if ((dir == CURSORUP) || (dir == CURSORDOWN))
        {
                horiz = FALSE;
                if (dir == CURSORUP) inc = FALSE;
                else inc = TRUE;

                GetDTAttrs(dto, DTA_TopVert, (IPTR)&val, TAG_DONE);
                top = (LONG)val;
                GetDTAttrs(dto, DTA_TotalVert, (IPTR)&val, TAG_DONE);
                total = (LONG)val;
                GetDTAttrs(dto, DTA_VisibleVert, (IPTR)&val, TAG_DONE);
                visible = (LONG)val;
        }
        else
        {
                horiz = TRUE;
                if (dir == CURSORLEFT) inc = FALSE;
                else inc = TRUE;

                GetDTAttrs(dto, DTA_TopHoriz, (IPTR)&val, TAG_DONE);
                top = (LONG)val;
                GetDTAttrs(dto, DTA_TotalHoriz, (IPTR)&val, TAG_DONE);
                total = (LONG)val;
                GetDTAttrs(dto, DTA_VisibleHoriz, (IPTR)&val, TAG_DONE);
                visible = (LONG)val;

        }

        oldtop = top;
        if (quali & (IEQUALIFIER_LALT | IEQUALIFIER_RALT | IEQUALIFIER_CONTROL))
        {
                if (inc) top = total;
                else top = 0;
        }
        else
                if (quali & (IEQUALIFIER_LSHIFT | IEQUALIFIER_RSHIFT))
                {
                        if (inc) top += visible - 1;
                        else top -= visible - 1;
                }
                else
                {
                        if (inc) top += delta;
                        else top -= delta;
                }

        if (top + visible > total) top = total - visible;
        if (top < 0) top = 0;

        if (top != oldtop)
        {
                struct Gadget *g;

                if (horiz)
                {
                        g = (struct Gadget *)gad[GAD_HORIZSCROLL];
                }
                else
                {
                        g = (struct Gadget *)gad[GAD_VERTSCROLL];
                }

                SetGadgetAttrs(g, win, NULL, PGA_Top, top,
                               TAG_DONE);

#ifdef __MORPHOS__
                /* Looks like setting PGA_Top on Amiga does not cause OM_NOTIFIEs
                   to be sent (to dto). Or something like that. */

                SetDTAttrs(dto, win, NULL, (horiz ? DTA_TopHoriz : DTA_TopVert), top, TAG_DONE);
#endif

        } /* if (top != oldtop) */

}

/*********************************************************************************************/

static void FitToWindow(void)
{
    if( pdt_fit_win )
    {
        int x, y;

        x = win->Width - (win->BorderLeft + win->BorderRight + 4);
        y = win->Height - (win->BorderTop + win->BorderBottom + 4);
        D(bug("=> width %ld height %ld\n", x, y));
        DoScaleMethod(x, y, pdt_keep_aspect);
//      DoLayout(TRUE); seems to be done by intuition ?
    }
}

/*********************************************************************************************/

static void HandleAll(void)
{
        struct TagItem      *tstate, *tags, *tag;
        struct MenuItem     *item;
        struct Gadget       *activearrowgad = NULL;
        WORD                arrowticker = 0, activearrowkind = 0;
        UBYTE               title[128];
        IPTR                tidata;
        ULONG               winmask;
        ULONG               printmask;
        UWORD               men;
        BOOL                quitme = FALSE;
        struct MsgPort      *printmp=NULL;
        union printerIO     *printio=NULL;
        struct Window       *printwin=NULL;
        const STRPTR        not_supported = "Sorry, not supported yet\n";

        winmask = 1UL << win->UserPort->mp_SigBit;

        while (!quitme)
        {
                ULONG mask, aboutmask, rcvd;

                aboutmask = GetAboutMask();
                printmask = printwin ? (1L << printwin->UserPort->mp_SigBit) : 0;
                mask = winmask | aboutmask | printmask;
                rcvd = Wait(mask | SIGBREAKF_CTRL_C);

                if (rcvd & SIGBREAKF_CTRL_C)
                        break;

                if ((rcvd & printmask) && printwin)
                {
                        if(SysReqHandler(printwin,NULL,FALSE)==0)
                        {
                          DoMethod(dto,DTM_ABORTPRINT,NULL);
                        }
                }

                if (mask & aboutmask)
                {
                        ProcessAbout();
                }

                if (mask & winmask)
                {
                        struct IntuiMessage *msg;

                        while((msg = (struct IntuiMessage *)GetMsg(win->UserPort)))
                        {
                                switch (msg->Class)
                                {
                                case IDCMP_REFRESHWINDOW:
                                        BeginRefresh(win);
                                        EndRefresh(win,TRUE);
                                        break;

                                case IDCMP_CLOSEWINDOW:
                                        quitme = TRUE;
                                        break;

                                case IDCMP_VANILLAKEY:
                                        switch(msg->Code)
                                        {
                                        case 27: /* ESC */
                                                quitme = TRUE;
                                                break;

                                        case 13: /* RETURN */
                                                if (dto_supports_search) DoTrigger(STM_SEARCH);
                                                else /*if (dto_supports_activate_field)*/ DoTrigger(STM_ACTIVATE_FIELD);
                                                RefreshDTObjects (dto, win, NULL, (IPTR) NULL);
                                                break;

                                        case 9: /* TAB */
                                                if (dto_supports_search_next) DoTrigger(STM_SEARCH_NEXT);
                                                else /*if (dto_supports_next_field)*/ DoTrigger(STM_NEXT_FIELD);
                                                break;

                                        case 8: /* BACKSPACE */
                                                if (dto_supports_retrace) DoTrigger(STM_RETRACE);
                                                break;

                                        case '>':
                                                if (dto_supports_browse_next) DoTrigger(STM_BROWSE_NEXT);
                                                break;

                                        case '<':
                                                if (dto_supports_browse_prev) DoTrigger(STM_BROWSE_PREV);
                                                break;

                                        case ' ': /* space -> PAGE DOWN */
                                                ScrollTo(CURSORDOWN, IEQUALIFIER_LSHIFT);
                                                break;

                                        } /* switch(msg->Code) */
                                        break;

                                case IDCMP_RAWKEY:
                                        switch(msg->Code)
                                        {
                                        case RAWKEY_NM_WHEEL_UP:
                                        case RAWKEY_NM_WHEEL_DOWN:
                                        case RAWKEY_NM_WHEEL_LEFT:
                                        case RAWKEY_NM_WHEEL_RIGHT:
                                        case CURSORUP:
                                        case CURSORDOWN:
                                        case CURSORRIGHT:
                                        case CURSORLEFT:
                                                ScrollTo(msg->Code, msg->Qualifier);
                                                break;

                                        case RAWKEY_HOME: /* HOME */
                                                if (dto_supports_contents && (msg->Qualifier & (IEQUALIFIER_LSHIFT | IEQUALIFIER_RSHIFT))) DoTrigger(STM_CONTENTS);
                                                else ScrollTo(CURSORUP, IEQUALIFIER_LALT);
                                                break;

                                        case RAWKEY_END: /* END */
                                                if (dto_supports_index && (msg->Qualifier & (IEQUALIFIER_LSHIFT | IEQUALIFIER_RSHIFT))) DoTrigger(STM_INDEX);
                                                else ScrollTo(CURSORDOWN, IEQUALIFIER_LALT);
                                                break;

                                        case RAWKEY_PAGEUP: /* PAGE UP */
                                                ScrollTo(CURSORUP, IEQUALIFIER_LSHIFT);
                                                break;

                                        case RAWKEY_PAGEDOWN: /* PAGE DOWN */
                                                ScrollTo(CURSORDOWN, IEQUALIFIER_LSHIFT);
                                                break;

                                        case RAWKEY_HELP: /* HELP */
                                                /*if (dto_supports_help)*/ DoTrigger(STM_HELP);
                                                break;

                                        case RAWKEY_CDTV_STOP: /* MMKEY STOP */
                                                /*if (dto_supports_stop)*/ DoTrigger(STM_STOP);
                                                break;

                                        case RAWKEY_CDTV_PLAY: /* MMKEY PLAY */
                                                /*if (dto_supports_play)*/ DoTrigger(STM_PLAY);
                                                break;

                                        case RAWKEY_CDTV_PREV: /* MMKEY PREV */
                                                if (dto_supports_browse_prev) DoTrigger(STM_BROWSE_PREV);
                                                break;

                                        case RAWKEY_CDTV_NEXT: /* MMKEY NEXT */
                                                if (dto_supports_browse_next) DoTrigger(STM_BROWSE_NEXT);
                                                break;

                                        case RAWKEY_CDTV_REW: /* MMKEY REW */
                                                /*if (dto_supports_rewind)*/ DoTrigger(STM_REWIND);
                                                break;

                                        case RAWKEY_CDTV_FF: /* MMKEY FF */
                                                /*if (dto_supports_fastforward)*/ DoTrigger(STM_FASTFORWARD);
                                                break;

                                        case 0x42: /* SHIFT TAB? */
                                                if (msg->Qualifier & (IEQUALIFIER_LSHIFT | IEQUALIFIER_RSHIFT))
                                                {
                                                        if (dto_supports_search_prev) DoTrigger(STM_SEARCH_PREV);
                                                        else /*if (dto_supports_prev_field)*/ DoTrigger(STM_PREV_FIELD);
                                                }
                                                break;
                                        } /* switch(msg->Code) */
                                        break;

                                case IDCMP_GADGETDOWN:
                                        arrowticker = 3;
                                        activearrowgad = (struct Gadget *)msg->IAddress;
                                        switch(activearrowgad->GadgetID)
                                        {
                                        case GAD_UPARROW:
                                                activearrowkind = CURSORUP;
                                                ScrollTo(CURSORUP, 0);
                                                break;

                                        case GAD_DOWNARROW:
                                                activearrowkind = CURSORDOWN;
                                                ScrollTo(CURSORDOWN, 0);
                                                break;

                                        case GAD_LEFTARROW:
                                                activearrowkind = CURSORLEFT;
                                                ScrollTo(CURSORLEFT, 0);
                                                break;

                                        case GAD_RIGHTARROW:
                                                activearrowkind = CURSORRIGHT;
                                                ScrollTo(CURSORRIGHT, 0);
                                                break;
                                        default:
                                                activearrowkind = 0;
                                                break;
                                        }
                                        break;

                                case IDCMP_INTUITICKS:
                                        if (activearrowkind)
                                        {
                                                if (arrowticker)
                                                {
                                                        arrowticker--;
                                                }
                                                else if (activearrowgad->Flags & GFLG_SELECTED)
                                                {
                                                        ScrollTo(activearrowkind, 0);
                                                }
                                        }
                                        break;

                                case IDCMP_GADGETUP:
                                        switch(((struct Gadget *)msg->IAddress)->GadgetID)
                                        {
                                        case GAD_UPARROW:
                                        case GAD_DOWNARROW:
                                        case GAD_LEFTARROW:
                                        case GAD_RIGHTARROW:
                                                activearrowkind = 0;
                                                break;
                                        }
                                        break;

                                case IDCMP_MENUPICK:
                                        men = msg->Code;
                                        while(men != MENUNULL)
                                        {
                                                if ((item = ItemAddress(menus, men)))
                                                {
                                                        switch((ULONG)GTMENUITEM_USERDATA(item))
                                                        {
                                                        case MSG_MEN_PROJECT_OPEN:
                                                                SetWindowPointer(win,WA_BusyPointer,TRUE,TAG_DONE);
                                                                filename = GetFile();
                                                                SetWindowPointer(win,WA_BusyPointer,FALSE,TAG_DONE);
                                                                if (filename) OpenDTO();
                                                                break;

                                                        case MSG_MEN_PROJECT_SAVEAS:
                                                                filename = GetFile();
                                                                if (filename) DoWriteMethod(filename, DTWM_RAW);
                                                                break;

                                                        case MSG_MEN_PROJECT_SAVEAS_IFF:
                                                                filename = GetFile();
                                                                if (filename) DoWriteMethod(filename, DTWM_IFF);
                                                                break;

                                                        case MSG_MEN_PROJECT_PRINT:
                                                                {
                                                                  if((printmp=CreateMsgPort()))
                                                                  {
                                                                     if((printio = (union printerIO *)CreateIORequest(printmp,sizeof(union printerIO))))
                                                                     {
                                                                       if(OpenDevice("printer.device",0,(struct IORequest *)printio,0)==0)
                                                                       {
                                                                         struct TagItem tags[2];
                                                                         struct dtPrint dtp;
                                                                         struct EasyStruct es;

#warning TODO: Is this possible without a previous DTM_PROCLAYOUT ?
                                                                         tags[0].ti_Tag     = DTA_RastPort;
                                                                         tags[0].ti_Data    = (ULONG)win->RPort;
                                                                         tags[1].ti_Tag     = TAG_DONE;

                                                                         dtp.MethodID       = DTM_PRINT;
                                                                         dtp.dtp_GInfo      = NULL;
                                                                         dtp.dtp_PIO        = printio;
                                                                         dtp.dtp_AttrList   = tags;

                                                                         es.es_StructSize   = sizeof(struct EasyStruct);
                                                                         es.es_Flags        = 0;
#warning TODO: add some locale strings here
                                                                         es.es_Title        = "Multiview Print";
                                                                         es.es_TextFormat   = "Printing ...";
                                                                         es.es_GadgetFormat = "Abort";

                                                                         if(PrintDTObjectA(dto,win,NULL,&dtp))
                                                                         {
                                                                           // launched successfully

                                                                           DB(("printjob started\n"));
                                                                           printwin = BuildEasyRequestArgs(win,&es,NULL,NULL);
                                                                         }
                                                                         else
                                                                         {
                                                                           DB(("printjob failed\n"));
                                                                           CloseDevice((struct IORequest *)printio);
                                                                           DeleteIORequest((struct IORequest*)printio);
                                                                           printio=NULL;
                                                                           DeleteMsgPort(printmp);
                                                                           printmp=NULL;  
                                                                         }

                                                                       }
                                                                       else
                                                                       {
                                                                         OutputMessage("failed to open printer device!");
                                                                         DB(("failed to open printer device\n"));
                                                                         DeleteIORequest((struct IORequest*)printio);
                                                                         printio=NULL;
                                                                         DeleteMsgPort(printmp);
                                                                         printmp=NULL;  
                                                                       }
                                                                     }
                                                                     else
                                                                     {
                                                                       DB(("failed to create printer io\n"));
                                                                       DeleteMsgPort(printmp);
                                                                       printmp=NULL;
                                                                     }
                                                                  }
                                                                }
                                                                break;

                                                        case MSG_MEN_PROJECT_ABOUT:
                                                                OpenAbout();
                                                                break;

                                                        case MSG_MEN_PROJECT_QUIT:
                                                                quitme = TRUE;
                                                                break;

                                                        case MSG_MEN_EDIT_MARK:
#if defined(_AROS) && !defined(__MORPHOS__)
                                                                if (StartDragSelect(dto))
#else
{
                                                                struct DTSpecialInfo *si;

                                                                /*
                                                                ** ClipView example on AmigaDev CD does just the following.
                                                                ** None of the checks AROS datatypes.library/StartDragSelect()
                                                                ** does.
                                                                */

                                                                si = (struct DTSpecialInfo *)(((struct Gadget *)dto)->SpecialInfo);
                                                                si->si_Flags |= DTSIF_DRAGSELECT;
                                                        }
#endif
                                                                {
#warning TODO: change mouse pointer to crosshair
                                                                }
                                                                break;

                                                        case MSG_MEN_EDIT_COPY:
                                                                {
                                                                        struct dtGeneral dtg;

                                                                        dtg.MethodID = DTM_COPY;
                                                                        dtg.dtg_GInfo = NULL;

                                                                        DoDTMethodA(dto, win, NULL, (Msg)&dtg);
                                                                }
                                                                break;

                                                        case MSG_MEN_EDIT_SELECTALL:
//                                                                OutputMessage(not_supported);
                                                                break;

                                                        case MSG_MEN_EDIT_CLEARSELECTED:
                                                                {
                                                                        struct dtGeneral dtg;

                                                                        dtg.MethodID = DTM_CLEARSELECTED;
                                                                        dtg.dtg_GInfo = NULL;

                                                                        DoDTMethodA(dto, win, NULL, (Msg)&dtg);
                                                                }
                                                                break;

                                                        case MSG_MEN_WINDOW_SEPSCREEN:
                                                                OutputMessage(not_supported);
                                                                break;

                                                        case MSG_MEN_WINDOW_MINIMIZE:
                                                                OutputMessage(not_supported);
                                                                break;

                                                        case MSG_MEN_WINDOW_NORMAL:
                                                                OutputMessage(not_supported);
                                                                break;

                                                        case MSG_MEN_WINDOW_MAXIMIZE:
                                                                OutputMessage(not_supported);
                                                                break;

                                                        case MSG_MEN_SETTINGS_SAVEDEF:
                                                                OutputMessage(not_supported);
                                                                break;

                                                        case MSG_MEN_PICT_ZOOM_IN:
                                                                pdt_zoom++;
                                                                if (pdt_zoom == -1 ) pdt_zoom = 1;
                                                                DoZoom(pdt_zoom);
                                                                break;

                                                        case MSG_MEN_PICT_ZOOM_OUT:
                                                                pdt_zoom--;
                                                                if (pdt_zoom == 0 ) pdt_zoom = -2;
                                                                DoZoom(pdt_zoom);
                                                                break;

                                                        case MSG_MEN_PICT_RESET:
                                                                pdt_zoom = 1;
                                                                DoZoom(pdt_zoom);
                                                                break;

                                                        case MSG_MEN_PICT_FIT_WIN:
                                                                pdt_fit_win = (item->Flags & CHECKED) ? TRUE : FALSE;
                                                                FitToWindow();
                                                                DoLayout(TRUE);
                                                                break;

                                                        case MSG_MEN_PICT_KEEP_ASPECT:
                                                                pdt_keep_aspect = (item->Flags & CHECKED) ? TRUE : FALSE;
                                                                FitToWindow();
                                                                DoLayout(TRUE);
                                                                break;

                                                        case MSG_MEN_PICT_FORCE_MAP:
                                                                SetDTAttrs (dto, NULL, NULL,
                                                                            PDTA_DestMode, (item->Flags & CHECKED) ? FALSE : TRUE,
                                                                            TAG_DONE);
                                                                DoLayout(TRUE);
                                                                break;

                                                        case MSG_MEN_PICT_DITHER:
                                                                SetDTAttrs (dto, NULL, NULL,
                                                                            PDTA_DitherQuality, (item->Flags & CHECKED) ? 4 : 0,
                                                                            TAG_DONE);
                                                                DoLayout(TRUE);
                                                                break;

                                                        case MSG_MEN_TEXT_WORDWRAP:
#if 0
                                                                if (item->Flags & CHECKED)
                                                                    D(bug("wordwrap enabled\n"));
                                                                else
                                                                    D(bug("wordwrap disabled\n"));
#endif
                                                                SetDTAttrs (dto, NULL, NULL,
                                                                            TDTA_WordWrap, (item->Flags & CHECKED) ? TRUE : FALSE,
                                                                            TAG_DONE);
                                                                DoLayout(TRUE);
                                                                break;

                                                        case MSG_MEN_TEXT_SEARCH:
                                                                if (dto_supports_search) DoTrigger(STM_SEARCH);
                                                                break;

                                                        case MSG_MEN_TEXT_SEARCH_PREV:
                                                                if (dto_supports_search_prev) DoTrigger(STM_SEARCH_PREV);
                                                                break;

                                                        case MSG_MEN_TEXT_SEARCH_NEXT:
                                                                if (dto_supports_search_next) DoTrigger(STM_SEARCH_NEXT);
                                                                break;

                                                        } /* switch(GTMENUITEM_USERDATA(item)) */

                                                        men = item->NextSelect;
                                                }
                                                else
                                                {
                                                        men = MENUNULL;
                                                }

                                        } /* while(men != MENUNULL) */
                                        break;

                                case IDCMP_NEWSIZE:
                                        D(bug("IDCMP NEWSIZE\n"));
                                        FitToWindow();
                                        break;

                                case IDCMP_IDCMPUPDATE:
                                        DB(("IDCMPUPDATE:\n"));
                                        tstate = tags = (struct TagItem *)msg->IAddress;
                                        while ((tag = NextTagItem ((struct TagItem **)&tstate)))
                                        {
                                                tidata = tag->ti_Data;
                                                DB(("attribute %lx=%lx\n", tag->ti_Tag, tidata));
                                                switch (tag->ti_Tag)
                                                {
                                                        /* Change in busy state */
                                                case DTA_Busy:
                                                        if (tidata)
                                                                SetWindowPointer (win, WA_BusyPointer, TRUE, WA_PointerDelay,TRUE,TAG_DONE);
                                                        else
                                                                SetWindowPointer (win, WA_Pointer, NULL, TAG_DONE);
                                                        DB(("MultiView: DTA_Busy = %ld\n", tidata));
                                                        break;

                                                case DTA_Title:
                                                        SetWindowTitles(win, (UBYTE *)tidata, (UBYTE *)~0);
                                                        DB(("MultiView: DTA_Title = \"%s\"\n", tidata));
                                                        break;

                                                        /* Error message */
                                                case DTA_ErrorLevel:
                                                        if (tidata)
                                                        {
                                                            ULONG errnum = GetTagData(DTA_ErrorNumber, NULL, tags);
                                                            STRPTR errstr = (STRPTR) GetTagData(DTA_ErrorString, (ULONG) "", tags);

                                                            if (errnum > 0 && errnum < DTERROR_UNKNOWN_DATATYPE)
                                                            {
                                                                UBYTE error[128];
                                                                sprintf(error, "Error: \"%s\" ", errstr);  
                                                                Fault(errnum, error, title, sizeof(title));
                                                                SetWindowTitles(win, title, (UBYTE *)~0);
                                                            } else if(errnum >= DTERROR_UNKNOWN_DATATYPE && errnum < DTMSG_TYPE_OFFSET)
                                                            {
                                                                sprintf(title, "Error: \"%s\" %s", errstr, GetDTString(errnum));
                                                                SetWindowTitles(win, title, (UBYTE *)~0);
                                                            }
                                                            /* PrintErrorMsg (errnum, (STRPTR) options[OPT_NAME]); */
                                                        }
                                                        break;

                                                        /* Time to refresh */
                                                case DTA_Sync:
#if 0
                                                        /* Refresh the DataType object */
                                                        RefreshDTObjects (dto, win, NULL, NULL);
#endif
                                                        break;
                                                case DTA_PrinterStatus:
                                                        /* Handle done printing here */
                                                        DB(("DTA_PrinterStatus %ld\n",tidata));

                                                        if(printio)
                                                        {
                                                           CloseDevice((struct IORequest *)printio);
                                                           DeleteIORequest((struct IORequest *)printio);
                                                           DeleteMsgPort(printmp);
                                                           printio=NULL;
                                                        }

                                                        if(printwin)
                                                        {
                                                           FreeSysRequest(printwin);
                                                           printwin=NULL;
                                                        }
                                                        break;

                                                } /* switch (tag->ti_Tag) */

                                        } /* while ((tag = NextTagItem ((const struct TagItem **)&tstate))) */
                                        break;

                                } /* switch (msg->Class) */

                                ReplyMsg((struct Message *)msg);

                        } /* while((msg = (struct IntuiMessage *)GetMsg(win->UserPort))) */
                } /* if (mask & winmask) */

        } /* while (!quitme) */

        if(printio)
        {
          /* make sure printio is aborted */

          DoMethod(dto,DTM_ABORTPRINT,(ULONG)NULL);
          CloseDevice((struct IORequest *)printio);
          DeleteIORequest((struct IORequest *)printio);
          DeleteMsgPort(printmp);
          printio=NULL;
        }

        CloseAbout();
}

/*********************************************************************************************/

int main(int argc, char **argv)
{
        InitLocale("Sys/multiview.catalog", 1);
        InitMenus(nm);
        InitMenus(nmpict);
        InitMenus(nmtext);
        OpenLibs();
        InitDefaults();
        GetArguments(argc, argv);
        LoadFont();
        MakeICObjects();
        OpenDTO();
        GetVisual();
        MakeGadgets();
        menus = MakeMenus(nm);
        pictmenus = MakeMenus(nmpict);
        textmenus = MakeMenus(nmtext);
        SetMenuFlags();
        MakeWindow();
        HandleAll();
        Cleanup(NULL);

        return 0;
}

/*********************************************************************************************/
